<?php

namespace App\Http\Controllers\Store;

use App\Http\Controllers\Controller;
use App\Models\Package;
use App\Models\Server;
use Illuminate\Http\Request;

class StoreController extends Controller
{
    public function index(Request $request)
    {
        // Get all active servers for the selection screen
        $servers = Server::where('is_active', true)->orderBy('name')->get();
        
        // Check if a server is selected
        $serverId = $request->get('server_id') ?? session('current_server_id');
        $server = null;
        
        if ($serverId) {
            $server = Server::where('is_active', true)->find($serverId);
            if ($server) {
                // Store in session for future requests
                session(['current_server_id' => $serverId]);
            }
        }
        
        // If no server is selected and we have servers, show server selection
        if (!$server && $servers->count() > 0) {
            return view('store.server-selection', compact('servers'));
        }
        
        // Build base query with server filter if specified
        $baseQuery = Package::where('is_active', true);
        
        if ($server) {
            $baseQuery->forServer($server->id);
        } else {
            // If no server specified, only show packages that are assigned to at least one server
            $baseQuery->withServers();
        }
        
        // Get all packages sorted by sort_order
        $packages = $baseQuery->orderBy('sort_order')->get();
        
        // Get unique categories for filtering
        $categories = $packages->pluck('type')->unique()->sort()->values();

        // Calculate statistics
        $stats = [
            'total_packages' => $packages->count(),
            'vip_packages' => $packages->where('type', 'vip')->count(),
            'money_packages' => $packages->where('type', 'money')->count(),
            'job_packages' => $packages->where('type', 'job')->count(),
        ];

        return view('store.index', compact(
            'packages',
            'categories',
            'servers',
            'server',
            'stats'
        ));
    }

    public function show($slug, Request $request)
    {
        $serverId = $request->get('server_id') ?? session('current_server_id');
        $server = null;
        
        if ($serverId) {
            $server = Server::where('is_active', true)->find($serverId);
        }
        
        $query = Package::where('slug', $slug)
            ->where('is_active', true);
            
        // If server is specified, ensure package is available on that server
        if ($server) {
            $query->forServer($server->id);
        } else {
            // If no server specified, ensure package is assigned to at least one server
            $query->withServers();
        }
        
        $package = $query->firstOrFail();
        
        return view('store.show', compact('package', 'server'));
    }

    /**
     * Switch to a different server
     */
    public function switchServer(Request $request, $serverId)
    {
        $server = Server::where('is_active', true)->findOrFail($serverId);
        session(['current_server_id' => $server->id]);
        
        return redirect()->route('store.index')->with('success', 'Switched to server: ' . $server->name);
    }

    /**
     * Clear server selection and return to server selection page
     */
    public function clearServer()
    {
        session()->forget('current_server_id');
        return redirect()->route('store.index');
    }

    public function tos()
    {
        return view('store.tos');
    }
}